%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This file replicates Figure B.I
%
% Karel Mertens and Jose Montiel-Olea, ``Marginal Tax Rates and Income, 
%                                    New Time Series Evidence''
% January, 2018
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
clear all; close all;
addpath('auxiliary files')

% Load Data
AMTR                = xlsread('data/DATA_MMO.xlsx','AMTR (Figure I)'); 
CONTROLS            = xlsread('data/DATA_MMO.xlsx','CONTROLS');
AINC                = xlsread('data/DATA_MMO.xlsx','LOG AVG INCOME');
NARR                = xlsread('data/DATA_MMO.xlsx','Narrative Shocks (Table IV)'); 
PROX                = NARR(:,6);
PROX(isnan(PROX))   = 0;

DATASET.TSERIES = [AMTR(:,1) -log(1-AMTR(:,2)) CONTROLS(:,2:8) AINC(:,2) PROX]; 
DATASET.LABEL   = {'DATES','LNAMTR1','RGDP','UNRATE','INFL','FFR','GOV_TOT','STPRICE','DEBT','AINC','PROX'};
DATASET.UNIT    = [0         1         1        2      2      2      1         1        1       1     2 ]; 
DATASET.FIGLABELS = [{'year' ,'1/(1-AMTR) All Tax Units','Real GDP','Unemployment Rate','Inflation', ...
                    'Federal Funds Rate','Government Spending','Real Stock Prices', 'Debt','Income (All Tax Units)','AMTR proxy'}];  
DATASET.MAP = containers.Map(DATASET.LABEL,1:size(DATASET.TSERIES,2));

% Specification
SPEC.sb              = 1946;  % Sample start year
SPEC.se              = 2012;  % Sample end year
TSERIES              = DATASET.TSERIES((DATASET.TSERIES(:,1)>=SPEC.sb)&(DATASET.TSERIES(:,1)<=SPEC.se),:);
SPEC.dates           = TSERIES(:,1);
SPEC.DET             = [SPEC.dates==1949 SPEC.dates==2008 ones(size(TSERIES,1),1)]; % Deterministic Terms (Put constant last)
SPEC.p               = 2; % Lag length
SPEC.NWlags          = 8; % Newey-West Lags
SPEC.irhor           = 6; % Length of IR horizon 
SPEC.select_variables= {'LNAMTR1','AINC','RGDP','UNRATE','INFL','FFR','GOV_TOT','STPRICE','DEBT'};
SPEC.vars            = TSERIES(:,cell2mat(values(DATASET.MAP,SPEC.select_variables)));
SPEC.taxshocks       = TSERIES(:,cell2mat(values(DATASET.MAP,{'PROX'})));
SPEC.MAP             = containers.Map([SPEC.select_variables],[1:size(SPEC.vars,2)]);   

% Estimate SVARIV model
SVARIV               = SPEC;
SVARIV               = doProxySVAR(SVARIV); 

% Inference:
clevel          = [95 68];    % CI Percentiles
nboot           = 10000;
SVARIVci          = doProxySVARci(SVARIV,clevel,0);
SVARIVci_wildbs   = doProxySVARci(SVARIV,clevel,1,nboot); 
SVARIVci_mswbs    = doProxySVARci(SVARIV,clevel,2,nboot);
SVARIVci_delta    = doProxySVARci(SVARIV,clevel,3);
SVARIVci_msw_wiv  = doProxySVARci(SVARIV,clevel,4);
SVARIVci_mbb      = doProxySVARci(SVARIV,clevel,5,nboot);

% Make Plot 
for j = [1 2]
    plotdisplay = {'AINC'};
    VARci.irsL  = SVARIVci_delta.irsL(:,:,j);
    VARci.irsH  = SVARIVci_delta.irsH(:,:,j);
    VARci.irsL2 = SVARIVci_msw_wiv.irsL(:,:,j);
    VARci.irsH2 = SVARIVci_msw_wiv.irsH(:,:,j);
    name = strcat('Figure_B_I_upper_',num2str(clevel(j)));
    legendlabels = {'Delta Method','MSW 2017 Weak IV-robust'};

    display1= cell2mat(values(SVARIV.MAP,plotdisplay));
        for nvar = 1:length(display1)
                f=figure;    
                box on; hold on 
                    plot(0:SVARIV.irhor-1,SVARIV.irs(:,display1(nvar)),'-','MarkerSize',4,'LineWidth',2,'Color', [0 0 0.5]);
                    p1=plot(0:SVARIV.irhor-1,VARci.irsH(:,display1(nvar)),'LineWidth',1,'Color', [0 0 0.5],'LineStyle','--');
                    plot(0:SVARIV.irhor-1,VARci.irsL(:,display1(nvar)),'LineWidth',1,'Color', [0 0 0.5],'LineStyle','--');
                    p2=plot(0:SVARIV.irhor-1,VARci.irsH2(:,display1(nvar)),'LineWidth',1,'Color', [0.9 0 0],'LineStyle','-.');
                    plot(0:SVARIV.irhor-1,VARci.irsL2(:,display1(nvar)),'LineWidth',1,'Color', [0.9 0 0],'LineStyle','-.');                    
                    hline(0,'k-')
                    ti = title(strcat(num2str(clevel(j)),'% Confidence Level'));
                    xl=xlabel('horizon (years)');
                    yl=ylabel('percent');
                    set(gca,'XTick',0:1:10)
                    axis([0 SVARIV.irhor-1 -0.5 3.5])
                    if j == 2
                    l = legend([p1,p2],legendlabels);              
                    set(l,'FontName', 'AvantGarde','FontSize',10,'Location','NorthWest','EdgeColor','white','box','off')
                    end
                    set([xl,yl], 'FontName', 'AvantGarde','FontSize',14);
                    set([ti], 'FontName', 'AvantGarde','FontSize',16);
                    str=strcat('figures/',name,'_',plotdisplay{nvar});
                    saveas(gcf,str,'epsc');
        end
    
end

for j = [1 2]
    plotdisplay = {'AINC'};
    VARci.irsL  = SVARIVci_delta.irsL(:,:,j);
    VARci.irsH  = SVARIVci_delta.irsH(:,:,j);
    VARci.irsL2 = SVARIVci_wildbs.irsL(:,:,j);
    VARci.irsH2 = SVARIVci_wildbs.irsH(:,:,j);
    VARci.irsL3 = SVARIVci_mswbs.irsL(:,:,j);
    VARci.irsH3 = SVARIVci_mswbs.irsH(:,:,j);
    VARci.irsL4 = SVARIVci_mbb.irsL(:,:,j);
    VARci.irsH4 = SVARIVci_mbb.irsH(:,:,j);
    name = strcat('Figure_B_I_lower_',num2str(clevel(j)));
    legendlabels = {'Delta Method','MR 2013 Wild Bootstrap','MSW 2017 Bootstrap','JL 2016 Bootstrap'};

    display1= cell2mat(values(SVARIV.MAP,plotdisplay));
        for nvar = 1:length(display1)
                f=figure;    
                box on; hold on 
                    plot(0:SVARIV.irhor-1,SVARIV.irs(:,display1(nvar)),'-','MarkerSize',4,'LineWidth',2,'Color', [0 0 0.5]);
                    p1=plot(0:SVARIV.irhor-1,VARci.irsH(:,display1(nvar)),'LineWidth',1,'Color', [0 0 0.5],'LineStyle','--');
                    plot(0:SVARIV.irhor-1,VARci.irsL(:,display1(nvar)),'LineWidth',1,'Color', [0 0 0.5],'LineStyle','--');
                    p2=plot(0:SVARIV.irhor-1,VARci.irsH2(:,display1(nvar)),'LineWidth',1,'Color', [0.9 0 0],'LineStyle','-.');
                    plot(0:SVARIV.irhor-1,VARci.irsL2(:,display1(nvar)),'LineWidth',1,'Color', [0.9 0 0],'LineStyle','-.');
                    p3=plot(0:SVARIV.irhor-1,VARci.irsH3(:,display1(nvar)),'LineWidth',1,'Color', [0 0 0],'LineStyle','--');
                    plot(0:SVARIV.irhor-1,VARci.irsL3(:,display1(nvar)),'LineWidth',1,'Color', [0 0 0],'LineStyle','--');
                    p4=plot(0:SVARIV.irhor-1,VARci.irsH4(:,display1(nvar)),'LineWidth',1,'Color', [0 0.7 0],'LineStyle',':');
                    plot(0:SVARIV.irhor-1,VARci.irsL4(:,display1(nvar)),'LineWidth',1,'Color', [0 0.7 0],'LineStyle',':');                    
                    hline(0,'k-')
                    ti = title(strcat(num2str(clevel(j)),'% Confidence Level'));
                    xl=xlabel('horizon (years)');
                    yl=ylabel('percent');
                    set(gca,'XTick',0:1:10)
                    axis([0 SVARIV.irhor-1 -0.5 5])
                    if j == 2
                    l = legend([p1,p2,p3,p4],legendlabels);              
                    set(l,'FontName', 'AvantGarde','FontSize',10,'Location','NorthWest','EdgeColor','white','box','off')
                    end
                    set([xl,yl], 'FontName', 'AvantGarde','FontSize',14);
                    set([ti], 'FontName', 'AvantGarde','FontSize',16);
                    str=strcat('figures/',name,'_',plotdisplay{nvar});
                    saveas(gcf,str,'epsc');
        end
    
end

